<?php
require_once __DIR__ . '/../utils/Database.php';
require_once __DIR__ . '/../utils/Helper.php';

class ShowsService {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    public function getShows($query = [], $onlyVisible = false) {
        $categoryId = $query['categoryId'] ?? null;
        $categories = $query['categories'] ?? null;
        
        $sql = "SELECT s.*, CONCAT_WS(', ', GROUP_CONCAT(c.name)) AS categories
                FROM shows s
                JOIN categories_shows cs ON s.id = cs.showId
                JOIN categories c ON cs.categoryId = c.id";
        
        $params = [];
        if ($categoryId) {
            $sql .= " WHERE c.id = ?";
            $params[] = $categoryId;
        } else if ($categories) {
            $sql .= " WHERE " . $this->getCategoriesCondition($categories);
        }
        
        $sql .= " GROUP BY s.id, s.name";
        
        if ($onlyVisible) {
            $sql .= " HAVING s.isVisible = 1";
        }
        
        $rows = $this->db->query($sql, $params);
        $results = Helper::emptyOrRows($rows);
        
        // Sort by showOrder
        usort($results, function($a, $b) {
            return $a['showOrder'] - $b['showOrder'];
        });
        
        return $results;
    }
    
    public function getGroupedShows() {
        $sql = "SELECT shows.*, categories.id AS categoryId, categories.name AS categoryName, 
                categories.categoryOrder, categories.thumbnailUrl AS categoryThumbnail
                FROM shows
                JOIN categories_shows ON shows.id = categories_shows.showId AND shows.isVisible = 1
                JOIN categories ON categories.id = categories_shows.categoryId AND categories.isVisible = 1
                WHERE shows.isVisible = 1";
        
        $rows = $this->db->query($sql);
        $data = Helper::emptyOrRows($rows);
        
        $formattedData = [];
        foreach ($data as $current) {
            $found = false;
            foreach ($formattedData as &$item) {
                if ($item['id'] === $current['categoryId']) {
                    $item['shows'][] = $current;
                    $found = true;
                    break;
                }
            }
            
            if (!$found) {
                $formattedData[] = [
                    'id' => $current['categoryId'],
                    'name' => $current['categoryName'],
                    'categoryOrder' => $current['categoryOrder'],
                    'categoryThumbnail' => $current['categoryThumbnail'],
                    'shows' => [$current]
                ];
            }
        }
        
        // Sort categories by order
        usort($formattedData, function($a, $b) {
            return $a['categoryOrder'] - $b['categoryOrder'];
        });
        
        // Sort shows within each category
        foreach ($formattedData as &$item) {
            usort($item['shows'], function($a, $b) {
                return $a['showOrder'] - $b['showOrder'];
            });
        }
        
        return $formattedData;
    }
    
    public function getShow($id) {
        $sql = "SELECT s.*, CONCAT_WS(', ', GROUP_CONCAT(c.name)) AS categories
                FROM shows s
                JOIN categories_shows cs ON s.id = cs.showId
                JOIN categories c ON cs.categoryId = c.id
                WHERE s.id = ?
                GROUP BY s.id, s.name";
        
        $rows = $this->db->query($sql, [$id]);
        return Helper::emptyOrObject($rows);
    }
    
    public function updateShow($show) {
        $connection = $this->db->getConnection();
        
        try {
            $connection->beginTransaction();
            
            // Update show
            $sql = "UPDATE shows 
                    SET name = ?, presenterName = ?, description = ?, showOrder = ?, isVisible = ?, thumbnailUrl = ?
                    WHERE id = ?";
            $stmt = $connection->prepare($sql);
            $stmt->execute([
                $show['name'],
                $show['presenterName'],
                $show['description'],
                $show['showOrder'],
                $show['isVisible'],
                $show['thumbnailUrl'],
                $show['id']
            ]);
            
            // Delete existing category associations
            $sql = "DELETE FROM categories_shows WHERE showId = ?";
            $stmt = $connection->prepare($sql);
            $stmt->execute([$show['id']]);
            
            // Get category IDs from names
            $categories = explode(',', $show['categories']);
            $placeholders = implode(',', array_fill(0, count($categories), '?'));
            $sql = "SELECT id FROM categories WHERE name IN ($placeholders)";
            $stmt = $connection->prepare($sql);
            $stmt->execute($categories);
            $categoryIds = $stmt->fetchAll();
            
            // Insert new category associations
            $sql = "INSERT INTO categories_shows (id, categoryId, showId) VALUES (NULL, ?, ?)";
            $stmt = $connection->prepare($sql);
            foreach ($categoryIds as $category) {
                $stmt->execute([$category['id'], $show['id']]);
            }
            
            $connection->commit();
            return true;
        } catch (Exception $e) {
            $connection->rollBack();
            throw $e;
        }
    }
    
    public function addShow($show) {
        $connection = $this->db->getConnection();
        
        try {
            $connection->beginTransaction();
            
            // Insert show
            $sql = "INSERT INTO shows (id, name, presenterName, description, showOrder, isVisible, thumbnailUrl)
                    VALUES(NULL, ?, ?, ?, ?, ?, ?)";
            $stmt = $connection->prepare($sql);
            $stmt->execute([
                $show['name'],
                $show['presenterName'],
                $show['description'],
                $show['showOrder'],
                $show['isVisible'],
                $show['thumbnailUrl']
            ]);
            $insertId = $connection->lastInsertId();
            
            // Get category IDs from names
            $categories = explode(',', $show['categories']);
            $placeholders = implode(',', array_fill(0, count($categories), '?'));
            $sql = "SELECT id FROM categories WHERE name IN ($placeholders)";
            $stmt = $connection->prepare($sql);
            $stmt->execute($categories);
            $categoryIds = $stmt->fetchAll();
            
            // Insert category associations
            $sql = "INSERT INTO categories_shows (id, categoryId, showId) VALUES (NULL, ?, ?)";
            $stmt = $connection->prepare($sql);
            foreach ($categoryIds as $category) {
                $stmt->execute([$category['id'], $insertId]);
            }
            
            $connection->commit();
            return $insertId;
        } catch (Exception $e) {
            $connection->rollBack();
            throw $e;
        }
    }
    
    public function deleteShow($id) {
        $connection = $this->db->getConnection();
        
        try {
            $connection->beginTransaction();
            
            $sql = "DELETE FROM shows WHERE id = ?";
            $stmt = $connection->prepare($sql);
            $stmt->execute([$id]);
            
            $sql = "DELETE FROM categories_shows WHERE showId = ?";
            $stmt = $connection->prepare($sql);
            $stmt->execute([$id]);
            
            $sql = "DELETE FROM program_live WHERE showId = ?";
            $stmt = $connection->prepare($sql);
            $stmt->execute([$id]);
            
            $sql = "DELETE FROM episodes WHERE showId = ?";
            $stmt = $connection->prepare($sql);
            $stmt->execute([$id]);
            
            $connection->commit();
            return true;
        } catch (Exception $e) {
            $connection->rollBack();
            throw $e;
        }
    }
    
    private function getCategoriesCondition($categories) {
        if (empty($categories)) {
            return " ";
        }
        
        $cats = explode(',', $categories);
        $conditions = array_map(function($cat) {
            return "c.name = '" . trim($cat) . "'";
        }, $cats);
        
        return implode(' OR ', $conditions);
    }
}

