<?php
require_once __DIR__ . '/../utils/Database.php';
require_once __DIR__ . '/../utils/Helper.php';

class EpisodesService {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    public function getEpisodes($onlyVisible = false) {
        $sql = "SELECT e.*, s.name AS showName
                FROM episodes e JOIN shows s ON e.showId = s.id";
        if ($onlyVisible) {
            $sql .= " WHERE e.isVisible = 1";
        }
        $sql .= " ORDER BY episodeOrder";
        
        $rows = $this->db->query($sql);
        return Helper::emptyOrRows($rows);
    }
    
    public function getEpisode($id) {
        $sql = "SELECT * FROM episodes WHERE id = ?";
        $rows = $this->db->query($sql, [$id]);
        return Helper::emptyOrObject($rows);
    }
    
    public function getEpisodesOfShow($id, $onlyVisible = false) {
        $sql = "SELECT e.*, s.name AS showName
                FROM episodes e JOIN shows s ON e.showId = s.id AND s.id = ?
                WHERE e.highlightOf IS NULL";
        if ($onlyVisible) {
            $sql .= " AND e.isVisible = 1";
        }
        $sql .= " ORDER BY e.episodeOrder";
        
        $rows = $this->db->query($sql, [$id]);
        return Helper::emptyOrRows($rows);
    }
    
    public function getEpisodesOfCategory($categoryId, $onlyVisible = false) {
        $sql = "SELECT e.* FROM episodes e 
                JOIN shows s ON e.showId = s.id 
                JOIN categories_shows cs ON s.id = cs.showId 
                WHERE cs.categoryId = ? AND e.highlightOf IS NULL";
        if ($onlyVisible) {
            $sql .= " AND e.isVisible = 1";
        }
        $sql .= " ORDER BY e.date DESC";
        
        $rows = $this->db->query($sql, [$categoryId]);
        return Helper::emptyOrRows($rows);
    }
    
    public function getLatestEpisodeOfShow($id, $onlyVisible = false) {
        $sql = "SELECT * FROM episodes 
                WHERE date IN (
                    SELECT MAX(date) FROM episodes 
                    WHERE highlightOf IS NULL";
        if ($onlyVisible) {
            $sql .= " AND isVisible = 1";
        }
        $sql .= " AND showId = ?)";
        
        $rows = $this->db->query($sql, [$id]);
        return Helper::emptyOrObject($rows);
    }
    
    public function getLatestEpisodeOfCategory($id, $onlyVisible = false) {
        $sql = "SELECT * FROM episodes 
                WHERE date IN (
                    SELECT MAX(date) FROM episodes e 
                    JOIN categories_shows cs ON e.showId = cs.showId 
                    WHERE e.highlightOf IS NULL";
        if ($onlyVisible) {
            $sql .= " AND e.isVisible = 1";
        }
        $sql .= " AND cs.categoryId = ?)";
        
        $rows = $this->db->query($sql, [$id]);
        return Helper::emptyOrObject($rows);
    }
    
    public function getLatest() {
        $sql = "SELECT * FROM episodes 
                WHERE date IN (
                    SELECT MAX(date) FROM episodes 
                    WHERE isVisible = 1 AND highlightOf IS NULL
                )";
        $rows = $this->db->query($sql);
        return Helper::emptyOrObject($rows);
    }
    
    public function getHighlights($id, $onlyVisible = false) {
        $sql = "SELECT * FROM episodes WHERE highlightOf = ?";
        if ($onlyVisible) {
            $sql .= " AND isVisible = 1";
        }
        
        $rows = $this->db->query($sql, [$id]);
        return Helper::emptyOrRows($rows);
    }
    
    public function addEpisode($data) {
        $sql = "INSERT INTO episodes 
                (id, title, description, episodeOrder, isVisible, thumbnailUrl, videoUrl, guest, videoDuration, showId, highlightOf, seasonId) 
                VALUES (NULL, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        $params = [
            $data['title'],
            $data['description'],
            $data['order'] ?? 0,
            $data['isVisible'],
            $data['thumbnailUrl'],
            $data['videoUrl'],
            $data['guest'] ?? null,
            $data['videoDuration'] ?? null,
            $data['showId'],
            $data['highlightOf'] ?? null,
            $data['seasonId'] ?? null
        ];
        return $this->db->execute($sql, $params);
    }
    
    public function updateEpisode($id, $data) {
        $sql = "UPDATE episodes 
                SET episodeOrder = ?, title = ?, isVisible = ?, thumbnailUrl = ?, videoUrl = ?, guest = ?, videoDuration = ?, showId = ?, highlightOf = ?, seasonId = ?, description = ?
                WHERE id = ?";
        $params = [
            $data['order'] ?? 0,
            $data['title'],
            $data['isVisible'],
            $data['thumbnailUrl'],
            $data['videoUrl'],
            $data['guest'] ?? null,
            $data['videoDuration'] ?? null,
            $data['showId'],
            $data['highlightOf'] ?? null,
            $data['seasonId'] ?? null,
            $data['description'],
            $id
        ];
        return $this->db->execute($sql, $params);
    }
    
    public function deleteEpisode($id) {
        $sql = "DELETE FROM episodes WHERE id = ?";
        return $this->db->execute($sql, [$id]);
    }
}

