# Telemaroc PHP Backend API

This is a PHP conversion of the Express.js backend for the Telemaroc mobile application. It provides the same REST API endpoints and works seamlessly with the existing mobile app.

## Features

- **Full API Compatibility**: All endpoints from the Express.js version are supported
- **MySQL Database**: Uses the same database schema
- **File Upload Support**: Image upload functionality with unique file naming
- **CORS Enabled**: Cross-origin requests are supported
- **Clean URLs**: Uses `.htaccess` for URL rewriting
- **Error Handling**: Comprehensive error handling with proper HTTP status codes
- **Security**: Protection for sensitive files and SQL injection prevention using PDO prepared statements

## Requirements

- PHP 7.4 or higher
- MySQL 5.7 or higher (or MariaDB 10.2+)
- cPanel hosting with:
  - Apache web server with `mod_rewrite` enabled
  - PDO MySQL extension
  - File upload permissions

## Directory Structure

```
server-php/
├── api/                    # API endpoint handlers
│   ├── categories.php
│   ├── channels.php
│   ├── episodes.php
│   ├── live.php
│   ├── search.php
│   ├── seasons.php
│   ├── shows.php
│   └── socialnetworks.php
├── environments/           # Environment configurations
│   ├── development.env
│   └── production.env
├── services/              # Business logic and database operations
│   ├── CategoriesService.php
│   ├── ChannelsService.php
│   ├── EpisodesService.php
│   ├── LiveService.php
│   ├── SearchService.php
│   ├── SeasonsService.php
│   ├── ShowsService.php
│   └── SocialNetworksService.php
├── utils/                 # Utility classes
│   ├── Database.php       # Database connection singleton
│   └── Helper.php         # Helper functions
├── storage/               # File upload directory (will be created automatically)
├── .htaccess             # Apache configuration
├── config.php            # Configuration loader
├── index.php             # Main entry point
└── README.md             # This file
```

## Installation on cPanel

### Step 1: Upload Files

1. Log in to your cPanel account
2. Navigate to **File Manager**
3. Go to your `public_html` directory (or a subdirectory if you want the API in a subfolder)
4. Create a new directory named `api` (or any name you prefer)
5. Upload all files from the `server-php` directory to this new directory

### Step 2: Configure Database

1. Ensure your MySQL database is already set up with the required tables
2. Update the `environments/production.env` file with your database credentials:

```env
DB_HOST=127.0.0.1
DB_PORT=3306
DB_USER=your_database_username
DB_PASSWORD=your_database_password
DB_NAME=your_database_name
CORS_ALLOWED_ORIGINS=*
```

**Important**: Replace the placeholders with your actual cPanel database credentials.

### Step 3: Set Permissions

1. In cPanel File Manager, right-click the `storage` directory
2. Set permissions to `755` or `777` (if 755 doesn't work)
3. Ensure the directory is writable by the web server

### Step 4: Update .htaccess (if needed)

If your API is in a subdirectory (e.g., `/api`), update the `RewriteBase` in `.htaccess`:

```apache
RewriteBase /api/
```

### Step 5: Test Your API

Visit your API URL in a browser:

```
https://yourdomain.com/api/
```

You should see:
```json
{"healthcheck":"ok"}
```

## API Endpoints

All endpoints from the original Express.js backend are supported:

### Categories
- `GET /categories` - Get all categories
- `GET /categories/:id` - Get category by ID
- `GET /categories?showId=X` - Get categories for a show
- `POST /categories/add` - Add new category
- `POST /categories/update/:id` - Update category
- `DELETE /categories/delete/:id` - Delete category

### Shows
- `GET /shows` - Get all shows
- `GET /shows/grouped` - Get shows grouped by categories
- `GET /shows/:id` - Get show by ID
- `POST /shows/show/add` - Add new show
- `POST /shows/update/:id` - Update show
- `DELETE /shows/delete/:id` - Delete show

### Episodes
- `GET /episodes` - Get all episodes
- `GET /episodes/:id` - Get episode by ID
- `GET /episodes/show/:id` - Get episodes of a show
- `GET /episodes/category/:id` - Get episodes of a category
- `GET /episodes/latest/all` - Get latest episode
- `GET /episodes/latest/:id` - Get latest episode of a show
- `GET /episodes/highlights/:id` - Get episode highlights
- `POST /episodes/add` - Add new episode
- `POST /episodes/update/:id` - Update episode
- `DELETE /episodes/delete/:id` - Delete episode

### Seasons
- `GET /seasons` - Get all seasons
- `GET /seasons/:id` - Get season by ID
- `GET /seasons/show/:id` - Get seasons of a show
- `GET /seasons/episodes/:id` - Get episodes of a season
- `POST /seasons/add` - Add new season
- `POST /seasons/update/:id` - Update season
- `DELETE /seasons/delete/:id` - Delete season

### Live
- `GET /live` - Get latest live link
- `GET /live/program` - Get today's live programs
- `GET /live/programs` - Get all live programs
- `GET /live/program/:id` - Get live program by ID
- `POST /live/add` - Add live program
- `POST /live/update` - Update live link
- `POST /live/program/update/:id` - Update live program
- `DELETE /live/delete/:id` - Delete live program

### Social Networks
- `GET /socialnetworks` - Get all social networks
- `GET /socialnetworks/:id` - Get social network by ID
- `POST /socialnetworks/add` - Add social network
- `POST /socialnetworks/update/:id` - Update social network
- `DELETE /socialnetworks/delete/:id` - Delete social network

### Channels
- `GET /channels` - Get all channels
- `GET /channels/:id` - Get channel by ID

### Search
- `GET /search?keyword=X` - Search shows and episodes

### Upload
- `POST /upload` - Upload an image file

## Updating Mobile App Configuration

Update your mobile app's API base URL to point to your new PHP backend:

**In `src/environments/environment.prod.ts`:**
```typescript
export const environment = {
  production: true,
  apiUrl: 'https://yourdomain.com/api'
};
```

## Troubleshooting

### Error: "Database connection failed"
- Check your database credentials in `environments/production.env`
- Ensure your database server is accessible
- Verify that PDO MySQL extension is enabled in PHP

### Error: "500 Internal Server Error"
- Check PHP error logs in cPanel (usually in `error_log` file)
- Ensure all file paths and includes are correct
- Verify PHP version is 7.4 or higher

### Upload not working
- Check `storage` directory permissions (should be 755 or 777)
- Verify upload limits in `.htaccess` or PHP configuration
- Check cPanel file upload limits

### Clean URLs not working
- Ensure `mod_rewrite` is enabled on your server
- Check `.htaccess` file is in the root directory of your API
- Verify `RewriteBase` is correctly set in `.htaccess`

### CORS errors from mobile app
- Update `CORS_ALLOWED_ORIGINS` in your `.env` file
- Or keep it as `*` to allow all origins (less secure but easier for development)

## Security Recommendations

1. **Change Database Password**: Use a strong, unique password for your database user
2. **Restrict CORS**: Set specific allowed origins instead of `*` in production
3. **Protect .env Files**: The `.htaccess` already blocks access to `.env` files
4. **Use HTTPS**: Always use HTTPS in production
5. **File Upload Validation**: Consider adding file type and size validation in the upload endpoint
6. **Regular Backups**: Set up automated backups for your database and files

## Support

If you encounter any issues, check:
1. PHP error logs in cPanel
2. Apache error logs
3. Browser console for API errors
4. Verify database connection and table structure

## Migration Notes

This PHP backend is a direct port of the Express.js backend and maintains:
- ✅ Same API endpoints
- ✅ Same request/response formats
- ✅ Same database schema
- ✅ Same business logic

**No changes required** in your mobile application!

